// Game State
const gameState = {
    mode: 'solo',
    operation: 'add',
    difficulty: 1,
    blueScore: 0,
    redScore: 0,
    soloScore: 0,
    targetScore: 10,
    currentProblem: { blue: null, red: null, solo: null },
    currentAnswer: { blue: '', red: '', solo: '' }
};

// Initialize game
window.addEventListener('DOMContentLoaded', () => {
    const playMode = localStorage.getItem('playMode') || 'solo';
    document.getElementById('gameMode').value = playMode;
    toggleTeamSetup();
});

// Toggle team setup visibility
function toggleTeamSetup() {
    const mode = document.getElementById('gameMode').value;
    const teamSetup = document.getElementById('teamSetup');

    if (mode === 'team') {
        teamSetup.style.display = 'block';
    } else {
        teamSetup.style.display = 'none';
    }
}

// Generate math problem
function generateProblem() {
    const diff = parseInt(document.getElementById('difficulty').value);
    let min, max;

    switch(diff) {
        case 1: min = 1; max = 10; break;
        case 2: min = 10; max = 50; break;
        case 3: min = 50; max = 150; break;
        case 4: min = 100; max = 1000; break;
    }

    const operation = document.getElementById('operationType').value;
    let ops = [];

    if (operation === 'mixed') {
        ops = ['add', 'subtract', 'multiply', 'divide'];
    } else {
        ops = [operation];
    }

    const selectedOp = ops[Math.floor(Math.random() * ops.length)];
    let num1, num2, answer, question, symbol;

    switch(selectedOp) {
        case 'add':
            num1 = Math.floor(Math.random() * (max - min + 1)) + min;
            num2 = Math.floor(Math.random() * (max - min + 1)) + min;
            answer = num1 + num2;
            symbol = '+';
            break;
        case 'subtract':
            num1 = Math.floor(Math.random() * (max - min + 1)) + min;
            num2 = Math.floor(Math.random() * (max - min + 1)) + min;
            if (num1 < num2) [num1, num2] = [num2, num1];
            answer = num1 - num2;
            symbol = '-';
            break;
        case 'multiply':
            const multMax = Math.floor(Math.sqrt(max));
            num1 = Math.floor(Math.random() * multMax) + 1;
            num2 = Math.floor(Math.random() * multMax) + 1;
            answer = num1 * num2;
            symbol = '×';
            break;
        case 'divide':
            num2 = Math.floor(Math.random() * 10) + 1;
            answer = Math.floor(Math.random() * 20) + 1;
            num1 = num2 * answer;
            symbol = '÷';
            break;
    }

    question = `${num1} ${symbol} ${num2} = ?`;
    return { question, answer };
}

// Display problem
function displayProblem(team) {
    const problem = generateProblem();
    gameState.currentProblem[team] = problem;
    gameState.currentAnswer[team] = '';

    const problemDiv = document.getElementById(`${team}Problem`);
    problemDiv.innerHTML = `
        <div class="problem-text">${problem.question}</div>
        <div class="answer-input" id="${team}Answer">_</div>
    `;
}

// Create number pad
function createNumberPad(team) {
    const keyboardDiv = document.getElementById(`${team}Keyboard`);
    const buttons = ['7', '8', '9', '4', '5', '6', '1', '2', '3', '-', '0', 'C'];

    keyboardDiv.innerHTML = '';

    buttons.forEach(btn => {
        const button = document.createElement('button');
        button.className = 'number-btn';
        if (btn === 'C') button.classList.add('clear-btn');
        button.textContent = btn;
        button.onclick = () => handleNumberInput(team, btn);
        keyboardDiv.appendChild(button);
    });

    const submitBtn = document.createElement('button');
    submitBtn.className = 'number-btn submit-btn';
    submitBtn.textContent = '✓ Javobni Yuborish';
    submitBtn.onclick = () => submitAnswer(team);
    keyboardDiv.appendChild(submitBtn);
}

// Handle number input
function handleNumberInput(team, value) {
    if (value === 'C') {
        gameState.currentAnswer[team] = '';
    } else if (value === '-') {
        if (gameState.currentAnswer[team] === '') {
            gameState.currentAnswer[team] = '-';
        }
    } else {
        gameState.currentAnswer[team] += value;
    }

    document.getElementById(`${team}Answer`).textContent = gameState.currentAnswer[team] || '_';
}

// Update race progress
function updateRaceProgress(team) {
    let score = 0;

    if (team === 'solo') {
        score = gameState.soloScore;
    } else if (team === 'blue') {
        score = gameState.blueScore;
    } else if (team === 'red') {
        score = gameState.redScore;
    }

    const percentage = (score / gameState.targetScore) * 100;
    const racer = document.getElementById(`${team}Racer`);
    const indicator = document.getElementById(`${team}Indicator`);

    racer.style.left = `${percentage}%`;
    indicator.textContent = `${Math.round(percentage)}%`;
}

// Submit answer
function submitAnswer(team) {
    const userAnswer = parseInt(gameState.currentAnswer[team]);
    const correctAnswer = gameState.currentProblem[team].answer;

    if (isNaN(userAnswer)) {
        alert('Iltimos, javobni kiriting!');
        return;
    }

    const problemDiv = document.getElementById(`${team}Problem`);

    if (userAnswer === correctAnswer) {
        problemDiv.classList.add('correct-answer');
        setTimeout(() => problemDiv.classList.remove('correct-answer'), 500);

        if (team === 'solo') {
            gameState.soloScore++;
        } else if (team === 'blue') {
            gameState.blueScore++;
        } else if (team === 'red') {
            gameState.redScore++;
        }

        updateRaceProgress(team);

        if (checkWinner()) {
            return;
        }

        displayProblem(team);
    } else {
        problemDiv.classList.add('wrong-answer');
        setTimeout(() => problemDiv.classList.remove('wrong-answer'), 500);
        gameState.currentAnswer[team] = '';
        document.getElementById(`${team}Answer`).textContent = '_';
    }
}

// Check winner
function checkWinner() {
    if (gameState.mode === 'solo') {
        if (gameState.soloScore >= gameState.targetScore) {
            setTimeout(() => showResult('solo'), 500);
            return true;
        }
    } else {
        if (gameState.blueScore >= gameState.targetScore) {
            setTimeout(() => showResult('blue'), 500);
            return true;
        }
        if (gameState.redScore >= gameState.targetScore) {
            setTimeout(() => showResult('red'), 500);
            return true;
        }
    }
    return false;
}

// Start game
function startGame() {
    const mode = document.getElementById('gameMode').value;
    const operation = document.getElementById('operationType').value;
    const difficulty = document.getElementById('difficulty').value;

    gameState.mode = mode;
    gameState.operation = operation;
    gameState.difficulty = difficulty;
    gameState.blueScore = 0;
    gameState.redScore = 0;
    gameState.soloScore = 0;

    if (mode === 'team') {
        const blueTeamName = document.getElementById('blueTeamName').value || "Ko'k Jamoa";
        const redTeamName = document.getElementById('redTeamName').value || "Qizil Jamoa";

        document.getElementById('blueTeamTitle').textContent = blueTeamName;
        document.getElementById('redTeamTitle').textContent = redTeamName;
        document.getElementById('blueLabel').textContent = blueTeamName;
        document.getElementById('redLabel').textContent = redTeamName;

        document.getElementById('soloGame').style.display = 'none';
        document.getElementById('teamGame').style.display = 'block';
        document.getElementById('soloLane').style.display = 'none';
        document.getElementById('blueLane').style.display = 'flex';
        document.getElementById('redLane').style.display = 'flex';

        displayProblem('blue');
        displayProblem('red');
        createNumberPad('blue');
        createNumberPad('red');
        updateRaceProgress('blue');
        updateRaceProgress('red');
    } else {
        document.getElementById('soloGame').style.display = 'block';
        document.getElementById('teamGame').style.display = 'none';
        document.getElementById('soloLane').style.display = 'flex';
        document.getElementById('blueLane').style.display = 'none';
        document.getElementById('redLane').style.display = 'none';

        displayProblem('solo');
        createNumberPad('solo');
        updateRaceProgress('solo');
    }

    document.getElementById('setupScreen').classList.remove('active');
    document.getElementById('gameScreen').classList.add('active');
}

// Show result
function showResult(winner) {
    const resultContent = document.getElementById('resultContent');
    let html = '';

    if (winner === 'solo') {
        html = `
            <div class="winner-emoji">🏆</div>
            <div class="winner-text">Tabriklaymiz! Poygada Yutdingiz!</div>
            <div class="result-stats">
                <div class="stat-row">
                    <span class="stat-label">To'plangan Ball:</span>
                    <span class="stat-value">${gameState.soloScore}</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">Maqsad:</span>
                    <span class="stat-value">${gameState.targetScore} ball</span>
                </div>
            </div>
        `;
    } else {
        const winnerName = winner === 'blue'
            ? document.getElementById('blueTeamTitle').textContent
            : document.getElementById('redTeamTitle').textContent;

        html = `
            <div class="winner-emoji">🎉</div>
            <div class="winner-text">${winnerName} G'OLIB!</div>
            <div class="result-stats">
                <div class="stat-row">
                    <span class="stat-label">${document.getElementById('blueTeamTitle').textContent}:</span>
                    <span class="stat-value">${gameState.blueScore} ball</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">${document.getElementById('redTeamTitle').textContent}:</span>
                    <span class="stat-value">${gameState.redScore} ball</span>
                </div>
            </div>
        `;
    }

    resultContent.innerHTML = html;
    document.getElementById('gameScreen').classList.remove('active');
    document.getElementById('resultScreen').classList.add('active');
}

// Restart game
function restartGame() {
    document.getElementById('resultScreen').classList.remove('active');
    document.getElementById('setupScreen').classList.add('active');
}

// Go home
function goHome() {
    window.location.href = '../index.php';
}

// Pause game
function pauseGame() {
    if (confirm('O\'yinni to\'xtatmoqchimisiz?')) {
        document.getElementById('gameScreen').classList.remove('active');
        document.getElementById('setupScreen').classList.add('active');
    }
}

// Show help
function showHelp() {
    document.getElementById('helpModal').classList.add('active');
}

// Close help
function closeHelp() {
    document.getElementById('helpModal').classList.remove('active');
}

// Close modal on ESC
document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape') {
        closeHelp();
    }
});

// Close modal on outside click
document.getElementById('helpModal').addEventListener('click', (e) => {
    if (e.target.id === 'helpModal') {
        closeHelp();
    }
});
