// Game State
const gameState = {
    mode: 'solo',
    operation: 'add',
    difficulty: 1,
    blueScore: 0,
    redScore: 0,
    soloScore: 0,
    ropePosition: 0, // -10 to +10
    targetScore: 7,
    currentProblem: { blue: null, red: null, solo: null },
    currentAnswer: { blue: '', red: '', solo: '' }
};

// Initialize game
window.addEventListener('DOMContentLoaded', () => {
    const playMode = localStorage.getItem('playMode') || 'solo';
    gameState.mode = playMode;

    if (playMode === 'team') {
        document.getElementById('teamSetup').style.display = 'block';
    } else {
        document.getElementById('teamSetup').style.display = 'none';
    }
});

// Generate math problem
function generateProblem() {
    const diff = parseInt(document.getElementById('difficulty').value);
    let min, max;

    switch(diff) {
        case 1: min = 1; max = 10; break;
        case 2: min = 10; max = 50; break;
        case 3: min = 50; max = 150; break;
        case 4: min = 100; max = 1000; break;
    }

    const operation = document.getElementById('operationType').value;
    let ops = [];

    if (operation === 'mixed') {
        ops = ['add', 'subtract', 'multiply', 'divide'];
    } else {
        ops = [operation];
    }

    const selectedOp = ops[Math.floor(Math.random() * ops.length)];
    let num1, num2, answer, question, symbol;

    switch(selectedOp) {
        case 'add':
            num1 = Math.floor(Math.random() * (max - min + 1)) + min;
            num2 = Math.floor(Math.random() * (max - min + 1)) + min;
            answer = num1 + num2;
            symbol = '+';
            break;
        case 'subtract':
            num1 = Math.floor(Math.random() * (max - min + 1)) + min;
            num2 = Math.floor(Math.random() * (max - min + 1)) + min;
            if (num1 < num2) [num1, num2] = [num2, num1];
            answer = num1 - num2;
            symbol = '-';
            break;
        case 'multiply':
            const multMax = Math.floor(Math.sqrt(max));
            num1 = Math.floor(Math.random() * multMax) + 1;
            num2 = Math.floor(Math.random() * multMax) + 1;
            answer = num1 * num2;
            symbol = '×';
            break;
        case 'divide':
            num2 = Math.floor(Math.random() * 10) + 1;
            answer = Math.floor(Math.random() * 20) + 1;
            num1 = num2 * answer;
            symbol = '÷';
            break;
    }

    question = `${num1} ${symbol} ${num2} = ?`;
    return { question, answer };
}

// Display problem
function displayProblem(team) {
    const problem = generateProblem();
    gameState.currentProblem[team] = problem;
    gameState.currentAnswer[team] = '';

    const problemDiv = document.getElementById(`${team}Problem`);
    problemDiv.innerHTML = `
        <div class="problem-text">${problem.question}</div>
        <div class="answer-input" id="${team}Answer"></div>
    `;
}

// Create number pad
function createNumberPad(team) {
    const keyboardDiv = document.getElementById(`${team}Keyboard`);
    const buttons = ['7', '8', '9', '4', '5', '6', '1', '2', '3', '-', '0', 'C'];

    keyboardDiv.innerHTML = '';

    buttons.forEach(btn => {
        const button = document.createElement('button');
        button.className = 'number-btn';
        if (btn === 'C') button.classList.add('clear-btn');
        button.textContent = btn;
        button.onclick = () => handleNumberInput(team, btn);
        keyboardDiv.appendChild(button);
    });

    const submitBtn = document.createElement('button');
    submitBtn.className = 'number-btn submit-btn';
    submitBtn.textContent = '✓ Javobni Yuborish';
    submitBtn.onclick = () => submitAnswer(team);
    keyboardDiv.appendChild(submitBtn);
}

// Handle number input
function handleNumberInput(team, value) {
    if (value === 'C') {
        gameState.currentAnswer[team] = '';
    } else if (value === '-') {
        if (gameState.currentAnswer[team] === '') {
            gameState.currentAnswer[team] = '-';
        }
    } else {
        gameState.currentAnswer[team] += value;
    }

    document.getElementById(`${team}Answer`).textContent = gameState.currentAnswer[team] || '_';
}

// Submit answer
function submitAnswer(team) {
    const userAnswer = parseInt(gameState.currentAnswer[team]);
    const correctAnswer = gameState.currentProblem[team].answer;

    if (isNaN(userAnswer)) {
        alert('Iltimos, javobni kiriting!');
        return;
    }

    const problemDiv = document.getElementById(`${team}Problem`);

    if (userAnswer === correctAnswer) {
        problemDiv.classList.add('correct-answer');
        setTimeout(() => problemDiv.classList.remove('correct-answer'), 500);

        if (team === 'solo') {
            gameState.soloScore++;
            document.getElementById('soloScore').textContent = gameState.soloScore;
            gameState.ropePosition += 2;
        } else if (team === 'blue') {
            gameState.blueScore++;
            document.getElementById('blueScore').textContent = gameState.blueScore;
            gameState.ropePosition -= 2;
        } else if (team === 'red') {
            gameState.redScore++;
            document.getElementById('redScore').textContent = gameState.redScore;
            gameState.ropePosition += 2;
        }

        updateRopePosition();
        checkWinner();

        if (!checkWinner()) {
            displayProblem(team);
            createNumberPad(team);
        }
    } else {
        problemDiv.classList.add('wrong-answer');
        setTimeout(() => problemDiv.classList.remove('wrong-answer'), 500);
        gameState.currentAnswer[team] = '';
        document.getElementById(`${team}Answer`).textContent = '_';
    }
}

// Update rope position
function updateRopePosition() {
    gameState.ropePosition = Math.max(-10, Math.min(10, gameState.ropePosition));
    const indicator = document.getElementById('ropeIndicator');
    const percentage = ((gameState.ropePosition + 10) / 20) * 100;
    indicator.style.left = `${percentage}%`;
}

// Check winner
function checkWinner() {
    if (gameState.mode === 'solo') {
        if (gameState.soloScore >= gameState.targetScore) {
            showResult('solo');
            return true;
        }
    } else {
        if (gameState.blueScore >= gameState.targetScore) {
            showResult('blue');
            return true;
        }
        if (gameState.redScore >= gameState.targetScore) {
            showResult('red');
            return true;
        }
    }
    return false;
}

// Start game
function startGame() {
    const operation = document.getElementById('operationType').value;
    const difficulty = document.getElementById('difficulty').value;

    gameState.operation = operation;
    gameState.difficulty = difficulty;
    gameState.blueScore = 0;
    gameState.redScore = 0;
    gameState.soloScore = 0;
    gameState.ropePosition = 0;

    if (gameState.mode === 'team') {
        const blueTeamName = document.getElementById('blueTeamName').value || "Ko'k Jamoa";
        const redTeamName = document.getElementById('redTeamName').value || "Qizil Jamoa";

        document.getElementById('blueTeamTitle').textContent = blueTeamName;
        document.getElementById('redTeamTitle').textContent = redTeamName;

        document.getElementById('soloGame').style.display = 'none';
        document.getElementById('teamGame').style.display = 'block';

        displayProblem('blue');
        displayProblem('red');
        createNumberPad('blue');
        createNumberPad('red');
    } else {
        document.getElementById('soloGame').style.display = 'block';
        document.getElementById('teamGame').style.display = 'none';

        displayProblem('solo');
        createNumberPad('solo');
    }

    updateRopePosition();

    document.getElementById('setupScreen').classList.remove('active');
    document.getElementById('gameScreen').classList.add('active');
}

// Show result
function showResult(winner) {
    const resultContent = document.getElementById('resultContent');
    let html = '';

    if (winner === 'solo') {
        html = `
            <div class="winner-emoji">🏆</div>
            <div class="winner-text">Tabriklaymiz!</div>
            <div class="result-stats">
                <div class="stat-row">
                    <span class="stat-label">To'plangan Ball:</span>
                    <span class="stat-value">${gameState.soloScore}</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">Arqon Pozitsiyasi:</span>
                    <span class="stat-value">${gameState.ropePosition > 0 ? '+' : ''}${gameState.ropePosition}</span>
                </div>
            </div>
        `;
    } else {
        const winnerName = winner === 'blue'
            ? document.getElementById('blueTeamTitle').textContent
            : document.getElementById('redTeamTitle').textContent;

        html = `
            <div class="winner-emoji">🎉</div>
            <div class="winner-text">${winnerName} G'OLIB!</div>
            <div class="result-stats">
                <div class="stat-row">
                    <span class="stat-label">${document.getElementById('blueTeamTitle').textContent}:</span>
                    <span class="stat-value">${gameState.blueScore} ball</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">${document.getElementById('redTeamTitle').textContent}:</span>
                    <span class="stat-value">${gameState.redScore} ball</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">Yakuniy Arqon Pozitsiyasi:</span>
                    <span class="stat-value">${gameState.ropePosition > 0 ? '+' : ''}${gameState.ropePosition}</span>
                </div>
            </div>
        `;
    }

    resultContent.innerHTML = html;
    document.getElementById('gameScreen').classList.remove('active');
    document.getElementById('resultScreen').classList.add('active');
}

// Restart game
function restartGame() {
    document.getElementById('resultScreen').classList.remove('active');
    document.getElementById('setupScreen').classList.add('active');
}

// Go home
function goHome() {
    window.location.href = '../index.php';
}

// Pause game
function pauseGame() {
    if (confirm('O\'yinni to\'xtatmoqchimisiz?')) {
        document.getElementById('gameScreen').classList.remove('active');
        document.getElementById('setupScreen').classList.add('active');
    }
}

// Show help
function showHelp() {
    document.getElementById('helpModal').classList.add('active');
}

// Close help
function closeHelp() {
    document.getElementById('helpModal').classList.remove('active');
}

// Close modal on ESC
document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape') {
        closeHelp();
    }
});

// Close modal on outside click
document.getElementById('helpModal').addEventListener('click', (e) => {
    if (e.target.id === 'helpModal') {
        closeHelp();
    }
});
